/***************************************************************
 * Name:      wxEDID_App.cpp
 * Purpose:   Code for Application Class
 * Author:    Tomasz Pawlak (tomasz.pawlak@wp.eu)
 * Copyright: Tomasz Pawlak (C) 2014-2025
 * License:   GPLv3+
 **************************************************************/

#include "wxEDID_App.h"
#include "wxEDID.h"

//(*AppHeaders
#include <wx/image.h>
//*)

#ifndef _RCD_AUTOGEN
   #include "config.h"
#else
   #define VERSION "1"
#endif

#include <wx/filename.h>
#include <wx/tokenzr.h>
#include <wx/log.h>

#include <wx/cmdline.h>

extern config_t config;

IMPLEMENT_APP(wxEDID_App);

static const wxString strFOpenErr = "[E!] Can't open file: '%s'\n";

bool wxEDID_App::OnInit() {
   bool wxsOK;

   wxsOK = CmdLineArgs();

   if (! wxsOK) return wxsOK;

   LoadConfig();

   //(* AppInitialize

   //wxInitAllImageHandlers();
   if ( wxsOK )
   {
      wxEDID_Frame* Frame = new wxEDID_Frame(0);
      Frame->Show();
      SetTopWindow(Frame);
   }
   //*)
   return wxsOK;
}

int wxEDID_App::OnExit() {

   SaveConfig();
   return true;
}

void wxEDID_App::SaveConfig() {
   static const wxString strCfgStruct =
   "dtd_keep_aspect: '%d'\n"
   "blk_dynamic_names: '%d'\n"
   "dta_grid_details: '%d'\n"
   "last_used_fpath: '%s'\n"
   "aui_layout: '%s'\n"
   "win_pos_x: '%d'\n"
   "win_pos_y: '%d'\n"
   "win_width: '%d'\n"
   "win_height: '%d'\n"
   "# File generated by wxEDID v" VERSION;

   wxString   cfg_str;
   wxFile     cfg_file;
   wxFileName cfg_fname;

   cfg_fname.SetPath( wxGetHomeDir() );
   cfg_fname.SetName( ".wxEDID_cfg" );
   cfg_str = cfg_fname.GetFullPath();

   if (! cfg_file.Open(cfg_str, wxFile::write) ) {
      wxLogStderr ErrLog;
      wxString    tmps;

      tmps.Printf(strFOpenErr, cfg_fname.GetFullPath());
      ErrLog.LogText(tmps);
   }

   cfg_str.Printf(strCfgStruct,
                  config.b_dtd_keep_aspect,
                  config.b_blk_name_dynamic,
                  config.b_dta_grid_details,
                  config.last_used_fpath,
                  config.aui_layout,
                  config.win_pos.x,
                  config.win_pos.y,
                  config.win_size.x,
                  config.win_size.y );

   cfg_file.Write(cfg_str);
}

void wxEDID_App::LoadConfig() {
   static const wxString strFTooBig  = "[E!] file too big: '%s'\n";

   wxString    cfg_str;
   wxFileName  cfg_fname;

   config.b_have_layout      = false;
   config.b_have_last_fpath  = false;
   config.b_dtd_keep_aspect  = true;
   config.b_dta_grid_details = false;
   config.b_blk_name_dynamic = true;

   cfg_fname.SetPath( wxGetHomeDir() );
   cfg_fname.SetName( ".wxEDID_cfg" );
   cfg_str = cfg_fname.GetFullPath();

   if (! wxFileExists(cfg_str)) return;

   {
      wxFile      cfg_file;
      wxString    tmps;
      wxLogStderr ErrLog;

      if (! cfg_file.Open(cfg_str, wxFile::read) ) {
         tmps.Printf(strFOpenErr, cfg_fname.GetFullPath());
         ErrLog.LogText(tmps);
         return;
      }

      if (cfg_file.Length() > 4096) {
         tmps.Printf(strFTooBig, cfg_fname.GetFullPath());
         ErrLog.LogText(tmps);
         return;
      }

      cfg_file.ReadAll(&cfg_str);
   }

   {
      long  tmpl;
      i32_t tkidx      = -1;
      u32_t layout_chk =  0;
      wxString          token;
      wxStringTokenizer tokenz(cfg_str, "'");

      while ( tokenz.HasMoreTokens() ) {
         token  = tokenz.GetNextToken();
         tkidx ++ ;
         if (0  == (tkidx & 1)) continue;
         if (18 <=  tkidx) return;
         //tokens @ odd idx hold the values
         switch (tkidx) {
            case 1: //dtd aspect ratio mode
               {
                  if (! token.ToLong(&tmpl)) return;
                  config.b_dtd_keep_aspect = (tmpl != 0);
                  break;
               }
            case 3: //use dynamic block names in block tree
               {
                  if (! token.ToLong(&tmpl)) return;
                  config.b_blk_name_dynamic = (tmpl != 0);
                  break;
               }
            case 5: //show field details in block data grid
               {
                  if (! token.ToLong(&tmpl)) return;
                  config.b_dta_grid_details = (tmpl != 0);
                  break;
               }
            case 7: //last path
               {
                  bool       dexist;
                  wxFileName last_dir(token);

                  cfg_str = last_dir.GetPath();
                  dexist  = wxDirExists( cfg_str );

                  if (dexist) {
                     config.last_used_fpath   = token;
                     config.b_have_last_fpath = true;
                  }
               }
               break;
            case 9: //AUI layout
               config.aui_layout  = token;
               layout_chk        ++ ;
               break;
            case 11: //win pos X
               if (! token.ToLong(&tmpl)) return;
               config.win_pos.x  = (int) tmpl;
               layout_chk       ++ ;
               break;
            case 13: //win pos Y
               if (! token.ToLong(&tmpl)) return;
               config.win_pos.y  = (int) tmpl;
               layout_chk       ++ ;
               break;
            case 15: //win width
               if (! token.ToLong(&tmpl)) return;
               config.win_size.x  = (int) tmpl;
               layout_chk        ++ ;
               break;
            case 17: //win height
               if (! token.ToLong(&tmpl)) return;
               config.win_size.y  = (int) tmpl;
               layout_chk        ++ ;
         }
      }

      config.b_have_layout = (layout_chk == 5);
   }

   return;
}

bool wxEDID_App::CmdLineArgs() {

  static const wxCmdLineEntryDesc sCmdLineArgs[] = {
       { wxCMD_LINE_SWITCH, "e" , "ignore-errors", NULL , wxCMD_LINE_VAL_NONE  , 0},
       { wxCMD_LINE_SWITCH, "w" , "all-writeable", NULL , wxCMD_LINE_VAL_NONE  , 0},
       { wxCMD_LINE_OPTION, "t" , "text"         , NULL , wxCMD_LINE_VAL_STRING, wxCMD_LINE_NEEDS_SEPARATOR},
       { wxCMD_LINE_PARAM , NULL, NULL           , NULL , wxCMD_LINE_VAL_STRING, wxCMD_LINE_PARAM_OPTIONAL },
       wxCMD_LINE_DESC_END
   };

   wxCmdLineParser cmd_parser(sCmdLineArgs, wxApp::argc, wxApp::argv);

   cmd_parser.Parse();

   config.b_cmd_bin_file   = false;
   config.b_cmd_ignore_err = cmd_parser.Found("e");
   config.b_cmd_ignore_rd  = cmd_parser.Found("w");
   config.b_cmd_txt_file   = cmd_parser.Found("t", &config.cmd_open_file_path);

   if (! config.b_cmd_txt_file) {
      int par_cnt = cmd_parser.GetParamCount();
      if (par_cnt > 0) {
         config.cmd_open_file_path = cmd_parser.GetParam(0);
      } else {
         return true;
      }
   }

   if (! wxFileExists(config.cmd_open_file_path)) {
      wxLogStderr ErrLog;
      wxString    tmps;

      tmps.Printf(strFOpenErr, config.cmd_open_file_path);
      ErrLog.LogText(tmps);

      return false;
   }

   config.b_cmd_bin_file = !config.b_cmd_txt_file;

   {
      wxFileName fname(config.cmd_open_file_path);

      if (! wxIsAbsolutePath(config.cmd_open_file_path) ) {

         fname.Normalize(wxPATH_NORM_DOTS|wxPATH_NORM_TILDE|wxPATH_NORM_ABSOLUTE, wxGetCwd());
      }

      config.cmd_open_file_path = fname.GetFullPath();
      config.last_used_fname    = fname.GetFullName();
   }

   return true;
}

